/******************************************************************
 *FILE: atmel_mxt_i2c.c
 *SW-COMPONENT: QNX atmel touch driver
 *DESCRIPTION: This source file is mainly used for i2c communications
 *between host and atmel touch slave.
 *COPYRIGHT: © 2019 Robert Bosch GmbH
 *
 *This program is free software; you can redistribute  it and/or modify it
 *under  the terms of  the GNU General  Public License as published by the
 *Free Software Foundation;  either version 2 of the  License, or (at your
 *option) any later version.
 ******************************************************************/
#include <string.h>
#include <malloc.h>
#include <stdint.h>
#include <errno.h>
#include <unistd.h>

#include "atmel_mxt_ts.h"
#include "atmel_mxt_i2c.h"

#ifndef UNITTEST
#include "i2c_client.h"
#else
#include "i2c_header.h"
#endif

int mxt_read_reg(int fd, uint16_t reg, unsigned len, void* buf)
{
	uint8_t write_buf[2];
	int ret, rc;
	bool retry = false;
	
	if (fd == -1)
		return -EINVAL;

	write_buf[0] = reg & 0xff;
	write_buf[1] = (reg >> 8) & 0xff;

retry_read:
	ret = i2c_combined_writeread(fd, write_buf, 2, buf, len);
	if (ret != len) {
		if (!retry) {
			rc = usleep(MXT_WAKEUP_TIME * 1000);
			if (EINTR != rc) {
				goto end;
			}

			retry = true;
			goto retry_read;
		} else {
			return -EIO;
		}
	}
end:
	return 0;
}

int mxt_write_reg(int fd, uint16_t reg, uint16_t len, const void *val)
{
	uint8_t *buf;
	int count;
	int ret;
	bool retry = false;
	
	if (fd == -1)
		return -EINVAL;

	count = len + 2;
	buf = malloc(count);
	if (!buf)
		return -ENOMEM;

	buf[0] = reg & 0xff;
	buf[1] = (reg >> 8) & 0xff;
	memcpy(&buf[2], val, len);

retry_write:
	ret = i2c_write(fd, buf, count);
	if (ret != count) {
		if (!retry) {
			ret = usleep(MXT_WAKEUP_TIME * 1000);
			if (EINTR != ret) {
				goto end;
			}

			retry = true;
			goto retry_write;
		} else {
			ret = -EIO;
		}
	} else {
		ret = 0;
	}
end:
	free(buf);
	return ret;	
}

int mxt_i2c_open(char* i2c_devname, unsigned slave_id, unsigned i2c_speed)
{	
	int fd;

	fd = i2c_open(i2c_devname);
	if (fd == -1)
		return -ENODEV;
	
	if((i2c_set_slave_addr(fd, slave_id, I2C_ADDRFMT_7BIT)) == -1)
		return -1;
	
	if((i2c_set_bus_speed(fd, I2C_SPEED_HIGH, &i2c_speed)) == -1)
		return fd; /* default speed is set,even on failure */

	return fd;
}

void mxt_i2c_close(int32_t fd)
{
	if (fd != -1)
		i2c_close(fd);

	fd = 0;
}

void get_atmel_libi2c_funcs(atmel_libi2c_funcs_t* funcs)
{
	funcs->open = mxt_i2c_open;
	funcs->close = mxt_i2c_close;
	funcs->read_reg = mxt_read_reg;
	funcs->write_reg = mxt_write_reg;
}
